/** 工具方法 ::: 
 *  @function produceNumber 随机数方法
 *  @function deepCopy 深拷贝方法
 */
import { deepCopy } from '@/utils/utils.js'

/** @name 模块组件编辑管理者混入
 *  @requires data {Object} domainEditProcess 编辑中模块数据 
 *  @requires data {Array} formList 编辑区表单节点列表
 *  @requires data {Array} domainList 分组列表
 *  @requires data {Object} formMap 表单项字典
 *  @requires data {Object} eventMap 事件字典
 *  @requires data {Object} formSettingState 整个表单样式配置
 *  @requires data {Object} activeNode 激活表单节点
 *  @requires data {Array} activeList 激活表单节点所在列表
 *  @requires mixin @/mixins/common/mix_make_config_manager.js
 *  @requires mixin @/mixins/common/mix_domain_manager.js
 *  @author 王汝鹏
 *  @version 2019-12-24
 */
export const mixDomainEditManager = {
  methods: {
    /** @name 分组组件编辑管理者
     *  @author 王汝鹏 
     *  @version 2019-12-22
     */
    domainEditManager() {
      let that = this;

      /** @name 判断是否为正在编辑的分组组件
       *  @param {Object} domain 分组组件
       *  @return {Boolean} 判断结果
       *  @author 王汝鹏
       *  @version 2019-12-22
       */
      const _isEqual = function(domain) {
        if (that.domainEditManager().getStatus() === 'normal') return false;
        let editDomainId = that.domainEditProcess.domain.node.id;
        let compareDomainId = domain.node.id;
        return editDomainId === compareDomainId;
      }

      /** @name 获取分组组件编辑状态
       *  @return {String} 编辑状态 { normal:"正常状态", editing:”修改中状态", changed:"已修改状态" }
       *  @author 王汝鹏
       *  @version 2019-12-22
       */
      const _getStatus = function() {
        return that.domainEditProcess.status;
      }

      /** @name 获取分组组件操作类型
       *  @return {String} 操作类型 { none:无操作, templates:模版操作, name:名称操作 }
       *  @author 王汝鹏
       *  @version 2020-01-05
       */
      const _getOperation = function() {
        return that.domainEditProcess.operation;
      }

      /** @name 获取编辑分组组件的名称
       *  @return {String} 分组组件名称
       *  @author 王汝鹏
       *  @version 2019-12-22
       */
      const _getDomainName = function() {
        if (that.domainEditManager().getStatus() === 'normal') return "";
        let domain = that.domainEditProcess.domain;

        return that.domainManager(domain).getName();
      }

      /** @name 设置分组组件的名称
       *  @param {String} name 分组组件名称
       *  @return {Boolean} 操作结果 
       *  @author 王汝鹏 
       *  @version 2020-01-05
       */
      const _setDomainName = function(name) {
        if (that.domainEditManager().getStatus() === 'normal') throw new Error("没有模块处于编辑状态");
        let domain = that.domainEditProcess.domain;

        that.domainManager(domain).setName(name);
        that.domainEditProcess.status = 'changed';
        return true;
      }

      /** @name 准备编辑分组组件
       *  @param {Object} domain 分组组件
       *  @param {String} operation 操作类型 { none:无操作, templates:模版操作, name:名称操作 }
       *  @return {Boolean} 操作结果
       *  @author 王汝鹏 
       *  @version 2019-12-22
       */
      const _edit = function(domain, operation) {
        if (that.domainEditManager().getStatus() === 'changed') throw new Error("有模块处于编辑状态，它的修改还没有保存");
        if (['templates', 'name'].includes(operation) === false) throw new Error(`分组组件操作类型不正确: operation=${operation};`);

        that.domainEditProcess.domain = domain;
        that.domainEditProcess.original = {
          formList: deepCopy(that.formList),
          domainList: deepCopy(that.domainList),
          formMap: deepCopy(that.formMap),
          eventMap: deepCopy(that.eventMap),
          formSettingState: deepCopy(that.formSettingState)
        };

        operation === 'templates'
          ? that.formList = that.domainManager(domain).getTemplates()
          : that.formList = [];

        that.domainEditProcess.operation = operation;
        that.domainEditProcess.status = 'editing';
        return true;
      }

      /** @name 更新分组组件模版数据
       *  @param {Object} templates 模版数据
       *  @return {Boolean} 操作结果
       *  @author 王汝鹏 
       *  @version 2019-12-22
       */
      const _setTemplates = function(templates) {
        if (that.domainEditManager().getStatus() === 'normal') throw new Error("没有模块处于编辑状态");
        if (that.domainEditManager().getOperation() !== 'templates') throw new Error("非模块组件模版编辑状态");

        let domain = that.domainEditProcess.domain;
        that.domainManager(domain).setTemplates(templates);
        that.domainEditProcess.status = 'changed';
        return true;
      }

      /** @name 提交分组组件改动
       *  @return {Boolean} 操作结果
       *  @author 王汝鹏
       *  @version 2019-12-22
       */
      const _submitChanged = function() {
        if (that.domainEditManager().getStatus() === 'normal') throw new Error("没有模块处于编辑状态");

        that.domainEditProcess.domain = null;
        that.domainEditProcess.original = {};
        that.domainEditProcess.status = 'normal';
        that.domainEditProcess.operation = "none";
        
        that.domainList = [...that.domainList];
        that.formList = [];
        that.activeNode = {};
        that.activeList = [];
        return true;
      }

      /** @name 回退到编辑前
       *  @return {Boolean} 操作结果
       *  @author 王汝鹏 
       *  @version 2019-12-22
       */
      const _rollbackChanged = function() {
        if (that.domainEditManager().getStatus() === 'normal') throw new Error("没有模块处于编辑状态");
        
        let { formList, domainList, formMap, eventMap, formSettingState } = that.domainEditProcess.original;
        that.formList = formList;
        that.domainList = domainList;
        that.formMap = formMap;
        that.eventMap = eventMap;
        that.formSettingState = formSettingState;

        that.domainEditProcess.domain = null;
        that.domainEditProcess.original = {};
        that.domainEditProcess.status = 'normal';
        that.domainEditProcess.operation = "none";
        return true;
      }

      /** @name 将编辑中的分组另存为其它分组
       *  @param {String} domainName 模块名称
       *  @return {Boolean} 操作结果
       *  @author 王汝鹏
       *  @version 2019-12-22
       */
      const _saveAs = function(domainName) {
        if (that.domainEditManager().getStatus() === 'normal') throw new Error("没有模块处于编辑状态");
        
        let copyList = that.formList.map(node => that.formNodeManager(node).copy());


        let { formList, domainList, formMap, eventMap, formSettingState } = that.domainEditProcess.original;
        that.formList = formList;
        that.domainList = domainList;
        that.formMap = { ...formMap, ...that.formMap };
        that.eventMap = eventMap;
        that.formSettingState = formSettingState;

        let domainConfig = that.makeDomainConfig(domainName, copyList);
        // 将新模块配置添加到模块组件列表
        that.domainList.push(domainConfig);

        that.domainEditProcess.domain = null;
        that.domainEditProcess.original = {};
        that.domainEditProcess.status = 'normal';
        that.domainEditProcess.operation = "none";
        return true;
      }

      return {
        isEqual(domain) { return _isEqual(domain); },
        getStatus() { return _getStatus(); },
        getOperation() { return _getOperation(); },
        edit(domain, operation) { return _edit(domain, operation); },
        getDomainName() { return _getDomainName(); },
        setDomainName(name) { return _setDomainName(name); },
        setTemplates(templates) { return _setTemplates(templates); },
        submitChanged() { _submitChanged(); },
        rollbackChanged() { _rollbackChanged(); },
        saveAs(domainName) { _saveAs(domainName); }
      }
    }
  }
}