

/** @name 事件配置管理者混入
 *  @requires data {Object} eventMap 事件字典
 *  @requires data {Object} formMap 表单项字典
 *  @author 王汝鹏
 *  @version 2019-12-24
 */
export const mixEventConfigManager = {
  methods: {
    /** @name 事件配置管理者
     *  @param {String} eventId 事件id
     *  @return {Object} 事件配置管理者
     *  @author 王汝鹏
     *  @version 2019-12-21
     */
    eventConfigManager(eventId) {
      let that = this;
      /** @name 删除关联触发表单项
       *  @param {String} formId 表单id
       *  @return {Boolean} 删除操作结果
       *  @author 王汝鹏
       *  @version 2019-12-21
       */
      const _deleteEmitRelated = function(formId) {
        let eventConfig = that.eventMap[eventId];
        if (!eventConfig) return false;
        let { emit, on } = eventConfig;
        
        eventConfig.emit = emit.filter(i => i.form !== formId);
        if (eventConfig.emit.length === 0) {
          let { form } = on;
          let formConfig = that.formMap[form];
          if (formConfig) formConfig.event.on = formConfig.event.on.filter(i => i !== eventId);
          Reflect.deleteProperty(that.eventMap, eventId);
        } 
        return true;
      }
      /** @name 删除事件
       *  @return {Boolean} 删除操作结果
       *  @author 王汝鹏
       *  @version 2019-12-21
       */
      const _deleteEvent = function() {
        let eventConfig = that.eventMap[eventId];
        if (!eventConfig) return false;
        let { emit, on } = eventConfig;
        
        for (let { form } of emit) {
          let formConfig = that.formMap[form];
          if (formConfig) formConfig.event.emit = formConfig.event.emit.filter(i => i !== eventId);
        }
        let { form } = on;
        let formConfig = that.formMap[form];
        if (formConfig) formConfig.event.on = formConfig.event.on.filter(i => i !== eventId);
        Reflect.deleteProperty(that.eventMap, eventId);
        return true;
      }

      /** @name 创建/更新事件配置
       *  @param {Object} event 新事件配置
       *  @return {Boolean} 操作结果
       *  @author 王汝鹏
       *  @version 2019-12-24
       */
      const _upsertEvent = function(event) {
        let {eventId, eventType, eventEmit, eventOn} = event;
        let oldEventConfig = that.eventMap[eventId];

        // 如果存在事件的历史配置，先删除历史配置
        if (oldEventConfig) that.eventConfigManager(eventId).deleteEvent();

        // 重新按照新的事件配置建立关系
        // 建立新事件关系
        // 遍历新事件配置中的 触发与监听 表单项
        // 找到表单项，建立与当前事件的关系
        for (let emitItem of eventEmit) {
          let emitForm = that.formMap[emitItem.form];
          emitForm.event.emit.push(eventId);
        }
        let onFrom = that.formMap[eventOn.form];
        onFrom.event.on.push(eventId);
        // 存储事件配置
        that.eventMap[eventId] = { 
          type: eventType, 
          emit: eventEmit, 
          on: eventOn 
        };
      }

      /** @name 获取配置区事件格式配置
       *  @return {Object} 表单配置区事件配置
       *  @author 王汝鹏
       *  @version 2019-12-24
       */
      const _transform = function(formId) {
        let { type, emit, on } = that.eventMap[eventId];
        let eventConfig = {
          eventId: eventId,        //当前事件id
          eventType: type,         //事件组合类型(并且、或者、...)
          triggerForm: formId,     //当前表单id
          triggerType: "change",   //当前表单项触发事件类型(值变化、点击、...)
          triggerValue: undefined, //触发事件的值
          emit: [],                //关联其它表单项触发事件配置
          on: {}                   //事件监听者配置
        };

        emit.forEach(e => {
          // 触发者配置: { form:表单项id, type:事件类型, value:触发事件的值 }
          let { form, type, value } = e;
          // 如果触发者为当前表单项
          // 更新当前表单项事件设置配置
          if (form === formId) {
            eventConfig.triggerForm = form;
            eventConfig.triggerType = type;
            eventConfig.triggerValue = value;
          // 其它触发者配置，添加到关联触发者配置中
          } else { eventConfig.emit.push({ form, type, value }); }
        })

        let { form, action } = on;
        eventConfig.on = { form, action };
        return eventConfig;
      }

      return {
        deleteEmitRelated(formId) { return _deleteEmitRelated(formId); },
        deleteEvent() { return _deleteEvent(); },
        upsertEvent(event) { return _upsertEvent(event); },
        transform(formId) { return _transform(formId); },
      }
    },
  }
}
