
/**************************************************************************************************************
 @name v-model工具方法混入 + v-model数据解构计算属性方法
 @description 主要方法说明
  混入方法说明:
    1.  在props中混入了value属性并限定类型为(Object|Array)
        !!! required 如果在父组件没有绑定 v-model 或 value ???
    2.1 methods :: valueRefresh 
        当父组件 v-model 或 value 值发生变化时会默认携带新值调用这个方法
        这里声明了一个空方法防止自动调用报错，如果想获取更新值需要在组件中声明同名方法覆盖
    2.2 methods :: refreshValue
        主动调用该方法，用于强制刷新父组件的数据，在必要的时候使用
    3.  在watch中混入value的监听, 值更新主动调用 valueRefresh 方法, 也可以在开发阶段在watch中检测值、调试值
  计算属性映射方法说明:
    功能效果: 根据配置生成 { example:{ get(){}, set(){} }, ... } 结构的数据
    参数说明: 
      key: 计算属性变量名称
      value :: parent: 用于查找计算属性变量父级的方法，方法参数为 v-model 或 value 绑定的值, 默认方法返回了 this.value
      value :: attr: 当前映射计算属性的字段名称, 会在 get、set 方法中操作parent方法返回数据中attr字段
      value :: getFormat: 在计算属性get方法返回值之前会调用getFormat方法对返回数据进行格式化后返回，默认方法没有做任何数据处理
      value :: setFormat: 在计算属性set方法设置值之前会调用setFormat方法对设置数据进行格式化后赋值，默认方法没有做任何数据处理
  @author 王汝鹏
  @version 2019-09-26
 **************************************************************************************************************/


/** @name 计算属性映射方法
 *  @description 使用例子:
 *    参数: { 
 *      example1:{ attr:"example1" }, 
 *      example2:{ parent:v=>v.e, attr:"example2", get:v => "hello "+v } 
 *    }
 *    结果: { 
 *      example1:{ get(){ return this.value['example1']; }, 
 *                 set(v){ this.value['example1']=v; this.$emit('input',{...this.value}); } } 
 *      example2:{ get(){ return 'hello ' + this.value.e['example2']; }, 
 *                 set(v){ this.value.e['example2']=v; this.$emit('input',{...this.value}); } }
 *    }
 *  @param {Object} items
 *  - {String} key 映射到计算属性中的计算属性名称
 *  - {Object} value 映射配置对象
 *    - {Function} parent 寻找属性父级方法，传入value变量，传出计算属性父级变量
 *    - {String} attr 变量名称
 *    - {Function} getFormat 读取数据格式化
 *    - {Function} setFormat 写入数据格式化
 *  @return {Object} 
 */
export const mapComputed = function(items) {
  let result = {};
  for (let key in items) {
    let defaultItem = {
      parent: i => i,
      attr: "",
      getFormat: function(val) { return val; },
      setFormat: function(val) { return val; }
    }
    let item = {...defaultItem, ...items[key]};
    result[key] = {
      get: function() {
        let parent = item.parent(this.value);
        let value = parent[item.attr];
        value = item.getFormat(value);
        return value;
      },
      set: function(newValue) {
        let parent = item.parent(this.value);
        parent[item.attr] = item.setFormat(newValue)
        this.$emit('input', {...this.value});
      }
    }
  }
  return result;
}

/** @name v-model 观察者
 *  @param {Object|Array} value
 *  @function valueRefresh value值发生更新钩子方法
 *  @function refreshValue 刷新value值
 *  @author 王汝鹏
 *  @version 2019-09-26
 */
export default {
  props: {
    // v-model接收
    value: {
      type: [Object,Array],
      required: true
    }
  },
  methods: {
    /** @name value值发生更新(值刷新))
     *  @author 王汝鹏
     *  @version 2019-09-26
     */
    valueRefresh(newValue) {},
    /** @name 刷新value值（刷新值）
     *  @author 王汝鹏
     *  @version 2019-09-26
     */
    refreshValue () {
      let model = (this.value instanceof Array) ? [...this.value] : {...this.value};
      this.$emit('input', model);
    }
  },
  watch: {
    /** @name value值观察者
     *  @description 调用value刷新钩子方法
     *  @author 王汝鹏
     *  @version 2019-09-26
     */
    value: {
      handler(val) { 
        this.valueRefresh(val) 
      },
      deep: true,
      immediate: true
    }
  }
}
